/*
 Abstract Syntax Tree (AST) Printer Example
 Design Tool Studio
 Taught by Patrick Hebron @ NYU ITP
 */

#pragma once

#include <iostream>
#include <string>
#include <memory>

typedef std::shared_ptr<class AstBase>		AstBaseRef;
typedef std::shared_ptr<class AstNumber>	AstNumberRef;
typedef std::shared_ptr<class AstBinaryAdd>	AstBinaryAddRef;

/** @brief abstract base class for AST */
class AstBase
{
  protected:
	
	/** @brief default constructor */
	AstBase();
	
  public:
	
	/** @brief virtual destructor */
	virtual ~AstBase();
	
	/** @brief pure virtual print method */
	virtual void print() const = 0;
	
	/** @brief pure virtual visitation method */
	virtual void accept(class Visitor& visitor) = 0;
};

/** @brief AST numeric entity */
class AstNumber : public AstBase
{
  private:
	
	float mValue;

	/** @brief value constructor */
	AstNumber(float value);
	
  public:
	
	/** @brief static creational method */
	template <typename ... Args> static AstNumberRef create(Args&& ... args)
	{
		return AstNumberRef( new AstNumber( std::forward<Args>( args )... ) );
	}
	
	/** @brief value getter method */
	const float& getValue() const;
	
	/** @brief print method */
	void print() const;
	
	/** @brief visitation method */
	void accept(Visitor& visitor);
};

/** @brief AST binary addition operator entity */
class AstBinaryAdd : public AstBase
{
  private:
	
	AstBaseRef mLhs, mRhs;
	
	/** @brief value constructor */
	AstBinaryAdd(AstBaseRef lhs, AstBaseRef rhs);
	
  public:
	
	/** @brief static creational method */
	template <typename ... Args> static AstBinaryAddRef create(Args&& ... args)
	{
		return AstBinaryAddRef( new AstBinaryAdd( std::forward<Args>( args )... ) );
	}
	
	/** @brief lhs getter method */
	AstBaseRef getLhs() const;
	
	/** @brief rhs getter method */
	AstBaseRef getRhs() const;
	
	/** @brief print method */
	void print() const;
	
	/** @brief visitation method */
	void accept(Visitor& visitor);
};
